/** @file   gameanims.h
 * @brief   Declarations of all the animations in the game.
 * @version $Revision: 1.3 $
 * @date    $Date: 2006/02/24 12:24:11 $
 * @author  Tomi Lamminsaari
 */

#ifndef H_WWW_GAMEANIMS_H
#define H_WWW_GAMEANIMS_H

#include <vector>
#include "weapon.h"
#include "www_exceptions.h"
#include "eng2d_Exception.h"

// Forward declarations.
namespace eng2d {
  class AnimContainer;
  class DataWarehouse;
  class Animation;
};



namespace WeWantWar {


/** @class  GameAnims
 * @brief   Creates and manages the animations of the game.
 * @author  Tomi Lamminsaari
 *
 * This class contains only static members.
 */
class GameAnims
{
public:

  ///
  /// Static members and methods
  /// ==========================
  
  
  /** Id codes for object animations.
   */
  enum PlaAnimId {
    /** Object is not doing anything. */
    EIdle = 0,
    /** Object is walking forward or backward. */
    EWalk,
    /** Object is strafing */
    EStrafe,
    /** Object is shooting/attacking but not moving. */
    EShoot,
    /** Object is shooting/attacking and moving forward/backward. */
    EShootWalk,
    /** Object is shooting/attacking and strafing. */
    EShootStrafe,
    /** Object is dying. */
    EDying,
    /** Object has been killed. */
    EKilled,
    /** Object is reloading its weapon. */
    EReload,
    /** Object is throwing */
    EThrowing,
    /** Object has no animation */
    ENoAnimation
  };
  
  
  
  /** Initializes the animations.
   * @return    0 if successful. Nonzero if fails.
   */
  static int init();
  
  /** Cleans up the animations.
   */
  static void cleanup();



  /** Returns the animation that represents the player with certain weapon.
   * 'id'.
   * @param     aWeapon           The weapon id. See Weapon - class.
   * @param     aAnimId           Animation id. See enum PlaAnimID.
   * @return    The animation.
   */
  static const eng2d::Animation& getAnimByWeapon( int aWeapon, int aAnimId );
  
  /** Returns the AnimItem that matches with given animation combine id.
   * @param   aCombineUid       Uid of the animation combine being returned.
   * @return  Pointer to AnimItem containing the requested item or nullpointer
   *          no animation combine mathces with the uid.
   */
  static const eng2d::AnimContainer* getAnimCombine( int aCombineUid );
  
  /** Finds the requested animation from warehouse.
   * @param   aCombineUid       Uid of the combine that contains the animation.
   * @param   aAnimIndex        Index of the animation within the combine. Default
   *                            value is 0 so if there is only one animation
   *                            then that gets returned.
   * @return  Reference to animation.
   * @exception Exception       Thrown if requested animation was not found.
   */
  static const eng2d::Animation& findAnimation( int aCombineUid, int aAnimIndex = 0 )
      throw( Exception, eng2d::xOutOfBounds );


  /** Constructors, destructor and operators **/

  GameAnims();
  ~GameAnims();
  

  
private:
  
  /** Pointer to warehouse that stores all the animations. */
  static eng2d::DataWarehouse* animationStore;

  /** Private methods **/
  

  GameAnims(const GameAnims& rO);
  GameAnims& operator = (const GameAnims& rO);

  /** Members **/

};


};  // end of namespace



#endif

/**
 * Version history
 * ===============
 * $Log: gameanims.h,v $
 * Revision 1.3  2006/02/24 12:24:11  lamminsa
 * no message
 *
 * Revision 1.2  2006/02/08 22:03:28  lamminsa
 * New animation and graphic system was implemented.
 *
 * Revision 1.1.1.1  2006/01/21 23:02:41  lamminsa
 * no message
 *
 * Revision 1.4  2006-01-10 22:55:07+02  lamminsa
 * decorative_chair2 added.
 *
 * Revision 1.3  2005-12-30 23:43:01+02  lamminsa
 * new barrel graphics.
 *
 * Revision 1.2  2005-11-23 00:25:58+02  lamminsa
 * Lightball alien and decorative object graphics were added.
 *
 * Revision 1.1  2005-11-13 14:35:04+02  lamminsa
 * RocketLauncher animations added.
 *
 */
 
